<?php

/*
Plugin Name: BOXIT WooCommerce Shipping 
Plugin URI:  https://touralbar.ir
Description: Adds a custom shipping method .
Version:     1.3
Author:      ALI MOUTABI ALAVI
Text Domain: my-custom-shipping
*/

// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Check if WooCommerce is active.
 * We only proceed if WC is loaded to prevent errors.
 */
if (in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {

    // 1. Register the custom shipping method class.
    add_filter('woocommerce_shipping_methods', 'register_my_custom_shipping_method');

    /**
     * Registers the custom shipping method class with WooCommerce.
     *
     * @param array $methods Available shipping methods.
     * @return array
     */
    function register_my_custom_shipping_method($methods)
    {
        // The key 'my_custom_method' must match the class ID defined inside the class.
        $methods['my_custom_method'] = 'WC_My_Custom_Shipping_Method';
        return $methods;
    }


    add_action('woocommerce_shipping_init', 'mcs_init_shipping_method');

    function mcs_init_shipping_method()
    {

        // 2. Define the custom shipping class.
        class WC_My_Custom_Shipping_Method extends WC_Shipping_Method
        {

            /**
             * Constructor.
             * @param int $instance_id Optional instance ID for multiple instances within a zone.
             */
            public function __construct($instance_id = 0)
            {
                $this->id = 'my_custom_method'; // Unique ID for your shipping method
                $this->instance_id = absint($instance_id);
                $this->method_title = __('boxit', 'text-domain');
                $this->method_description = __('boxit shipping method ', 'text-domain');

                // Supports features like zones and instance settings.
                $this->supports = array(
                    'shipping-zones',
                    'instance-settings',
                    'instance-settings-modal',
                );

                // Initialize settings and properties.
                $this->init_form_fields(); // Defines the form fields.
                $this->init_settings();    // Loads the saved settings from the DB.

                // Get instance properties (loaded via init_settings).
                $this->title = $this->get_option('title');
                // Crucially, retrieve the 'enabled' setting here.
                //$this->enabled = $this->get_option( 'enabled' ); 

                // Save settings in admin if submitted.
                add_action('woocommerce_update_options_shipping_' . $this->id, array($this, 'process_admin_options'));
            }

            /**
             * Defines the settings form fields for this shipping method.
             */
            public function init_form_fields()
            {
                $this->instance_form_fields = array(
                    // This 'enabled' field automatically creates the Enable/Disable toggle in the admin UI.
                    // 'enabled' => array(
                    //     'title'   => __( 'Enable/Disable', 'text-domain' ),
                    //     'type'    => 'checkbox',
                    //     'label'   => __( 'Enable this shipping method', 'text-domain' ),
                    //     'default' => 'yes', // Default to enabled.
                    // ),
                    'title' => array(
                        'title' => __('Method Title', 'text-domain'),
                        'type' => 'text',
                        'description' => __('Title shown to the customer during checkout.', 'text-domain'),
                        'default' => __('باکسیت', 'text-domain'),
                        'desc_tip' => true,
                    ),
                    'cost' => array(
                        'title' => __('Cost', 'text-domain'),
                        'type' => 'text',
                        'description' => __('Enter the fixed cost for this delivery method.', 'text-domain'),
                        'default' => '0',
                        'desc_tip' => true,
                        'placeholder' => '0',
                    ),
                );
            }

            /**
             * Calculates the shipping cost and adds a rate.
             *
             * @param array $package Package data.
             */
            public function calculate_shipping($package = array())
            {
                // Check the 'enabled' setting before calculating and adding a rate.
                if ('yes' !== $this->enabled) {
                    return;
                }

                ### Accessing Customer Location Data ###

                // State (e.g., 'CA', 'NY')
                $state_code = $package['destination']['state'];

                // City (e.g., 'Los Angeles', 'New York')
                $city = $package['destination']['city'];

                // Postcode (e.g., '90210')
                $postcode = $package['destination']['postcode'];

                // Country (e.g., 'US')
                $country_code = $package['destination']['country'];

                if ('تبریز' !== $city) {
                    //return;
                }

                // Retrieve the cost setting, defaulting to 0 if not set.
                $cost = floatval($this->get_option('cost', 0));

                if ($cost == 0) {
                    $this->title = $this->title . ' : ' . 'پس کرایه';
                }

                $rate = array(
                    'id' => $this->get_rate_id(),
                    'label' => $this->title, // Uses the title setting from the admin UI
                    'cost' => $cost,
                    'taxable' => 'no', // Set 'yes' if the cost is taxable
                );

                // Add the rate to WooCommerce.
                $this->add_rate($rate);
            }
        }
    }
}


add_action('woocommerce_order_status_processing', 'send_order_to_external_site_curl_once_checked');
add_action('woocommerce_order_status_completed', 'send_order_to_external_site_curl_once_checked');

function send_order_to_external_site_curl_once_checked($order_id)
{
    $order = wc_get_order($order_id);
    if (!$order)
        return;

    // Check if already sent successfully (1) or failed (3) to avoid multiple sends
    $sent_status = $order->get_meta('_sent_to_external');
    if ($sent_status == 1){
        return; // already sent successfully
    }
    

    // Check if My Custom Shipping was selected
    //var_dump( $order->get_shipping_method());
    $shipping_methods = $order->get_shipping_methods();
    $found_custom = false;
    $method_name = '';
    foreach ($shipping_methods as $shipping_item) {
        if ($shipping_item->get_method_id() === 'my_custom_method') {
            $found_custom = true;
            $method_name = $shipping_item->get_method_title();
            break;
        }
    }
    if (!$found_custom)
        return; // exit if customer didn't choose our shipping

    
    // Get shipping state/province name
    $states = WC()->countries->get_states($order->get_shipping_country());
    $province_name = $states[$order->get_shipping_state()] ?? $order->get_shipping_state();

    // Prepare data using shipping info
    $data = [
        'site_name'   => get_bloginfo('name') . ' - ' . home_url(),
        'person_name' => $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name(),
        'address'     => $province_name 
                        . ' ' . $order->get_shipping_city() 
                        . ' ' . $order->get_shipping_address_1() 
                        . ' ' . $order->get_shipping_address_2(),
        'postal_code' => $order->get_shipping_postcode(),
        'order_id'    => $order->get_id(),
        'mobile'      => $order->get_billing_phone(), // usually phone stays billing
        'method'      => $method_name,
        'status'      => 'completed',
        'token'       => '12345', // match your insert.php
    ];

    //cURL request
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://touralbar.ir/insert.php');
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

    $response = curl_exec($ch);
    //var_dump($response);


    if (curl_errno($ch)) {
        error_log('cURL Error: ' . curl_error($ch));
        $order->update_meta_data('_sent_to_external', 3); // failed
        $order->save();
    } else {
        error_log('Order sent via cURL: ' . $response);

        // Check if response contains “Inserted successfully”
        if (strpos($response, 'Inserted successfully') !== false) {
            $order->update_meta_data('_sent_to_external', 1); // success

            //send note
            $order->add_order_note('data send to BOXIT shipping service.'.'  RE='.$response);
        } else {
            $order->update_meta_data('_sent_to_external', 3); // failed

            //send note
            $order->add_order_note(' Failed to send order to BOXIT shipping service.'.'  RE= '.$response);
        }
        $order->save();
    }

    curl_close($ch);
}


add_action('woocommerce_before_thankyou', 'show_external_send_error_notice');

function show_external_send_error_notice($order_id) {
    if (!$order_id) return;

    $order = wc_get_order($order_id);
    if (!$order) return;

    $sent_status = $order->get_meta('_sent_to_external');

    if ($sent_status == 3) {
        wc_add_notice(
            __('مشکلی در ارسال اطلاعات به باکسیت به وجود امده است لطفا این مورد رو به ادمین سایت اطلاع دهید .', 'woocommerce'),
            'error'
        );
    }
}